﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using Microsoft.VisualBasic;
using NDI.SLIKDA.Interop;

namespace OPCServer_Simulator
{
    public partial class Form1 : Form
    {
        public Form1()
        {
            InitializeComponent();
        }

        private ISLIKTags myOpcTags;
        private Random randomizer = new Random();

        private void Form1_Load(object sender, EventArgs e)
        {
            // create a new instance of an empty tag database
            myOpcTags = SlikServer1.SLIKTags;

            // define a simple read/write access-right, saves us OR'ing this
            // enumeration multiple times
            AccessPermissionsEnum readWriteAccess = AccessPermissionsEnum.sdaReadAccess | AccessPermissionsEnum.sdaWriteAccess;


            // A Tag name is "just" a name. But when you prefix that name with a '.'
            // then the OPC Server toolkit will automatically place that tag within
            // a group. This simplifies the OPC Client experience when they browse
            // your OPC Servers address-space.
            // In this case, we will defined two groups:
            //  (a) USER
            //  (b) SIMULATION
            // .... you will see these are prefixes to the tagnames.

            // add the user-tags. These are the tags that represent the 3 form
            // controls on the LEFT side of the form
            myOpcTags.Add("User.CheckBox", (int)readWriteAccess, 0, 0, DateTime.Now, null);
            myOpcTags.Add("User.Numeric", (int)readWriteAccess, 0, 0, DateTime.Now, null);
            myOpcTags.Add("User.TextBox", (int)readWriteAccess, 0, 0, DateTime.Now, null);

            // add the simulation tags. These are the tags that represent the 3
            // form controls on the RIGHT side of the form
            myOpcTags.Add("Simulation.ProgressBar", (int)readWriteAccess, 0, 0, DateTime.Now, null);
            myOpcTags.Add("Simulation.Numeric", (int)readWriteAccess, 0, 0, DateTime.Now, null);
            myOpcTags.Add("Simulation.CheckBox", (int)readWriteAccess, 0, 0, DateTime.Now, null);

            // setup the tag DATA TYPES of these tags. We could've done these above,
            // but we've separated them here to illustrate the 2 tag groupings above.
            myOpcTags["User.CheckBox"].DataType = (short)VariantType.Boolean;
            myOpcTags["User.Numeric"].DataType = (short)VariantType.Integer;
            myOpcTags["User.TextBox"].DataType = (short)VariantType.String;
            myOpcTags["Simulation.ProgressBar"].DataType = (short)VariantType.Integer;
            myOpcTags["Simulation.Numeric"].DataType = (short)VariantType.Integer;
            myOpcTags["Simulation.CheckBox"].DataType = (short)VariantType.Boolean;
            
            // In this example, we will also register and start the server
            // Under normal circumstances you would NOT do this, because this is a
            // step that should be done once and once-only, at the end of an
            // installation routine. Most OPC Server implementations allow the use
            // of a command-line argument to call this function.

            SlikServer1.RegisterServer();

             // Now START the OPC Server interface. This is called whenever your
             // application starts.
            SlikServer1.StartServer();
        }

        private void ExitButton_Click(object sender, EventArgs e)
        {
            if ((MessageBox.Show("Close the OPC Server and Exit?", "Exit?", MessageBoxButtons.YesNo, MessageBoxIcon.Question, MessageBoxDefaultButton.Button2) == System.Windows.Forms.DialogResult.Yes))
            {
                try
                {
                    // Notify the OPC Clients that you want to shut down. This gives
                    // each OPC Client enough time to gracefully disconnect.
                    SlikServer1.RequestDisconnect("User Requested Shutdown");

                    // This function, just like the ".RegisterServer" as seen in the 
                    // Form_Load() event is a function that should not be called under
                    // normal circumstances. This should be done by the installer
                    // when the application is being removed from the computer.
                    //SlikServer1.UnregisterServer()
                }
                catch (Exception ex)
                {
                    Debug.WriteLine(ex.Message);
                }
                finally
                {
                    Application.Exit();
                }
            }

        }

        private void NumericUpDown1_ValueChanged(object sender, EventArgs e)
        {
            timer1.Interval = (int)NumericUpDown1.Value;
        }

        private void SimulationEnabled_CheckedChanged(object sender, EventArgs e)
        {
            timer1.Enabled = SimulationEnabled.Checked;
        }

        
        private void SimulationCheckbox_CheckedChanged(object sender, EventArgs e)
        {
            if (myOpcTags != null)
                myOpcTags["Simulation.CheckBox"].SetVQT(SimulationCheckbox.Checked, 192, DateTime.Now);
        }

        private void SimulationNumeric_ValueChanged(object sender, EventArgs e)
        {
            if( myOpcTags != null)
            {
                ISLIKTag tag  = myOpcTags["Simulation.Numeric"];
                tag.SetVQT(Convert.ToInt32(SimulationNumeric.Value), 192, DateTime.Now);
            }
        }

        private void UserCheckBox_CheckedChanged(object sender, EventArgs e)
        {
           if( myOpcTags != null)
           {
               myOpcTags["User.Numeric"].SetVQT(Convert.ToInt32(UserNumeric.Value), 192, DateTime.Now);
           }
        }

        private void UserNumeric_ValueChanged(object sender, EventArgs e)
        {
            if (myOpcTags != null)
            {
                myOpcTags["User.Numeric"].SetVQT(Convert.ToInt32(UserNumeric.Value), 192, DateTime.Now);
            }
        }

        private void UserTextBox_TextChanged(object sender, EventArgs e)
        {
            if (myOpcTags != null)
            {
                myOpcTags["User.TextBox"].SetVQT(UserTextBox.Text, 192, DateTime.Now);
            }
        }

        private void timer1_Tick(object sender, EventArgs e)
        {
            SimulationProgressBar.Value = Convert.ToInt32(randomizer.Next(100));
            SimulationNumeric.Value = Convert.ToDecimal(randomizer.Next(99999));
            SimulationCheckbox.Checked = Convert.ToBoolean(randomizer.Next(-1, 1));

            myOpcTags["Simulation.ProgressBar"].SetVQT(Convert.ToInt32(SimulationProgressBar.Value), 192, DateTime.Now);
        }

        private void SlikServer1_OnClientConnect(object sender, SLIKServer.OnClientConnectEventArgs eventArgs)
        {
            ConnectedClients.Value = eventArgs.NumClients;
        }

        private void SlikServer1_OnClientDisconnect(object sender, SLIKServer.OnClientDisconnectEventArgs eventArgs)
        {
            ConnectedClients.Value = eventArgs.NumClients;
        }

        private void SlikServer1_OnWrite(object sender, SLIKServer.OnWriteEventArgs eventArgs)
        {
            for (int i = 0; i <= (eventArgs.Count - 1); i++)
            {
                // Get the "next" item in the list
                ISLIKTag currentItem = eventArgs.Tags[i];
                object currentValue = eventArgs.Values[i];

                // We will look at the name of the OPC Tag to then figure out WHERE
                // to SEND the data to....
                switch ((currentItem.Name))
                {

                    // Checkbox on the LEFT side of the form
                    case "User.CheckBox":
                        UserCheckBox.Checked = Convert.ToBoolean(currentValue);

                        break;

                    // Numeric control on the LEFT side of the form
                    case "User.Numeric":
                        UserNumeric.Value = Convert.ToInt32(currentValue);

                        break;

                    // Textbox on the LEFT side of the form
                    case "User.TextBox":
                        UserTextBox.Text = Convert.ToString(currentValue);

                        break;

                    // Progress bar on the RIGHT side of the form
                    case "Simulation.ProgressBar":
                        SimulationProgressBar.Value = Convert.ToInt32(currentValue);

                        break;

                    // Numeric control on the RIGHT side of the form
                    case "Simulation.Numeric":
                        SimulationNumeric.Value = Convert.ToInt32(currentValue);

                        break;

                    // Checkbox on the RIGHT side of the form
                    case "Simulation.CheckBox":
                        SimulationCheckbox.Checked = Convert.ToBoolean(currentValue);
                        break;
                }

                // Specify that the Item at *this* position is NOT in error!
                eventArgs.Errors[i] = (int) OPCDAErrorsEnum.sdaSOK;
            }

            // Now specify that we completed this event successfully
            eventArgs.Result = (int) OPCDAErrorsEnum.sdaSOK;

        }

        private void SlikServer1_OnRead(object sender, SLIKServer.OnReadEventArgs eventArgs)
        {
            //
            // Iterate thru each/every item that the OPC Client has requested us to READ

            {

                for (int i = 0; i <= (eventArgs.Count - 1); i++)
                {
                    // Get the "next" item in the list
                    ISLIKTag currentItem = eventArgs.Tags[i];

                    // Now to return the value of this tag, along with
                    // GOOD quality, and a current timestamp
                    //   (Quality codes are: 192=GOOD; 0=BAD)
                    //
                    // We will look at the name of the OPC Tag to then figure out WHERE
                    // to get the data from....
                    switch ((currentItem.Name))
                    {
                        // Checkbox on the LEFT side of the form
                        case "User.CheckBox":
                            eventArgs.Tags[i].SetVQT(UserCheckBox.Checked, 192, DateTime.Now);

                            break;

                        // Numeric control on the LEFT side of the form
                        case "User.Numeric":
                            eventArgs.Tags[i].SetVQT(Convert.ToInt32(UserNumeric.Value), 192, DateTime.Now);

                            break;

                        // Textbox on the LEFT side of the form
                        case "User.TextBox":
                            eventArgs.Tags[i].SetVQT(UserTextBox.Text, 192, DateTime.Now);

                            break;

                        // Progress bar on the RIGHT side of the form
                        case "Simulation.ProgressBar":
                            eventArgs.Tags[i].SetVQT(SimulationProgressBar.Value, 192, DateTime.Now);

                            break;

                        // Numeric control on the RIGHT side of the form
                        case "Simulation.Numeric":
                            eventArgs.Tags[i].SetVQT(Convert.ToInt32(SimulationNumeric.Value), 192, DateTime.Now);

                            break;

                        // Checkbox on the RIGHT side of the form
                        case "Simulation.CheckBox":
                            eventArgs.Tags[i].SetVQT(SimulationCheckbox.Checked, 192, DateTime.Now);
                            break;
                    }

                    // Specify that the Item at *this* position is NOT in error!
                    eventArgs.Errors[i] = (int)OPCDAErrorsEnum.sdaSOK;
                }

                // Now specify that we completed this event successfully
                eventArgs.Result = (int)OPCDAErrorsEnum.sdaSOK;

            }


        }

       

        
    }
}
